// Timer Module
const timer = {
    currentGame: null,
    currentLevel: 0,
    timeRemaining: 0,
    isPaused: true,
    intervalId: null,
    blindStructure: null,
    
    // Start the timer
    start(game) {
        console.log('Timer.start() called with game:', game);
        
        if (!game || !game.blindLevels || game.blindLevels.length === 0) {
            console.error('Invalid game structure passed to timer:', game);
            alert('Error: Blind structure not configured properly');
            return;
        }
        
        this.currentGame = game;
        this.currentLevel = 0;
        this.blindStructure = game.blindLevels;
        this.isPaused = true; // Start paused - user clicks "Start" to begin
        
        console.log('Blind structure loaded:', this.blindStructure);
        
        // Set pause button to show "Start" since we're paused
        const btnText = document.getElementById('pause-resume-text');
        if (btnText) btnText.textContent = 'Start';
        const btn = document.getElementById('btn-pause-resume');
        if (btn) btn.style.background = '#4CAF50'; // Green for "ready to start"
        
        // Initialize first level (but don't start countdown yet)
        this.nextLevel();
    },
    
    // Move to next blind level
    nextLevel() {
        if (!this.blindStructure || this.currentLevel >= this.blindStructure.length) {
            console.log('Tournament complete - no more levels');
            this.stopCountdown();
            return;
        }
        
        this.currentLevel++;
        const level = this.blindStructure[this.currentLevel - 1];
        this.timeRemaining = level.duration * 60; // Convert to seconds
        
        console.log(`Moving to level ${this.currentLevel}: ${level.sb}/${level.bb}, ${level.duration} min`);
        
        this.updateDisplay();
        
        // Only auto-start countdown if not paused (i.e., during level transitions mid-game)
        if (!this.isPaused) {
            this.startCountdown();
        }
    },
    
    // Start countdown
    startCountdown() {
        // Clear any existing interval first
        this.stopCountdown();
        
        console.log(`Starting countdown with ${this.timeRemaining} seconds remaining, isPaused: ${this.isPaused}`);
        
        if (this.isPaused) {
            return; // Don't start if paused
        }
        
        this.intervalId = setInterval(() => {
            if (this.isPaused) {
                return; // Skip tick if paused
            }
            
            this.timeRemaining--;
            
            if (this.timeRemaining <= 0) {
                this.nextLevel();
            } else {
                this.updateDisplay();
            }
        }, 1000);
        
        console.log('Countdown interval started, ID:', this.intervalId);
    },
    
    // Stop countdown
    stopCountdown() {
        if (this.intervalId) {
            clearInterval(this.intervalId);
            this.intervalId = null;
        }
    },
    
    // Update display
    updateDisplay() {
        const currentLevelData = this.blindStructure[this.currentLevel - 1];
        const nextLevelData = this.currentLevel < this.blindStructure.length ? 
            this.blindStructure[this.currentLevel] : null;
        
        // Update blinds
        document.getElementById('current-blinds').textContent = 
            `${currentLevelData.sb}/${currentLevelData.bb}`;
        
        if (nextLevelData) {
            document.getElementById('next-blinds').textContent = 
                `${nextLevelData.sb}/${nextLevelData.bb}`;
        } else {
            document.getElementById('next-blinds').textContent = 'Final Level';
        }
        
        // Update time
        const minutes = Math.floor(this.timeRemaining / 60);
        const seconds = this.timeRemaining % 60;
        document.getElementById('time-remaining').textContent = 
            `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
        
        // Update level
        document.getElementById('current-level').textContent = this.currentLevel;
        document.getElementById('total-levels').textContent = this.blindStructure.length;
        
        // Change color if time is running out
        const timeDisplay = document.getElementById('time-remaining');
        if (this.timeRemaining <= 60) {
            timeDisplay.style.color = '#f44336'; // Red
        } else if (this.timeRemaining <= 180) {
            timeDisplay.style.color = '#ff9800'; // Orange
        } else {
            timeDisplay.style.color = '#4CAF50'; // Green
        }
    },
    
    // Toggle pause
    togglePause() {
        this.isPaused = !this.isPaused;
        console.log('Toggle pause, now isPaused:', this.isPaused);
        
        const btn = document.getElementById('btn-pause-resume');
        const btnText = document.getElementById('pause-resume-text');
        
        if (this.isPaused) {
            if (btnText) btnText.textContent = 'Resume';
            if (btn) btn.style.background = '#ff9800'; // Orange for paused
        } else {
            if (btnText) btnText.textContent = 'Pause';
            if (btn) btn.style.background = '#f44336'; // Red for running
            // Start the countdown when unpausing
            this.startCountdown();
        }
    },
    
    // Adjust time
    adjustTime(minutes) {
        this.timeRemaining += minutes * 60;
        if (this.timeRemaining < 0) this.timeRemaining = 0;
        this.updateDisplay();
    },
    
    // Skip to next level
    skipLevel() {
        if (confirm('Skip to next blind level?')) {
            this.nextLevel();
        }
    },
    
    // Save state (for pausing game)
    getState() {
        return {
            currentLevel: this.currentLevel,
            timeRemaining: this.timeRemaining,
            isPaused: this.isPaused,
            blindStructure: this.blindStructure
        };
    },
    
    // Restore state (for resuming game)
    setState(state) {
        if (!state) return;
        
        this.currentLevel = state.currentLevel || 0;
        this.timeRemaining = state.timeRemaining || 0;
        this.isPaused = state.isPaused !== undefined ? state.isPaused : true;
        this.blindStructure = state.blindStructure || null;
        
        // Update button state
        const btnText = document.getElementById('pause-resume-text');
        const btn = document.getElementById('btn-pause-resume');
        
        if (this.isPaused) {
            if (btnText) btnText.textContent = 'Resume';
            if (btn) btn.style.background = '#ff9800';
        } else {
            if (btnText) btnText.textContent = 'Pause';
            if (btn) btn.style.background = '#f44336';
            this.startCountdown();
        }

        this.updateDisplay();
    }
};

// Make timer globally available
window.timer = timer;
